/*
** 100902 PHUNSY 2650 micro computer emulator
** 100925 for the phunsy 1 cpu cycle time is 3s
** 101026
** 101113 added mdcr read data clock activity while cassette in reverse
** 101114 changed exit by ~ key to DEL key
** 101114 added extended i/o
** 101114 added print to logfile
** 101120 SDL added
** 101121 bug in Q bank memory movement corrected
** 101121 memory filled with HALT instructions
** 101123 indirect addressing bug fixed
** 101124 added exit on monitor or mdcr rom write
** 101127 phunsy roms included in exe
** 101127 arrow keys added
** 101127 added 4 cassette feature
** 101201 small error in phgraphpix[128] corrected
** 140104 texts simulator changed to emulator
** 230505 re-worked the code to work with SDL2.
** 230509 added Fullscreen mode (use -F commandline option)
**        added GreenScreen mode (use -G commandline option)
**
** NOTE: the 2650B is implemented except for the changes in instruction cycles.
*/
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <SDL2/SDL.h>
#include "version.h"


#define noMDCRDEBUG


#include "roms.h"

static const uint8_t hello[160] = {
  0x06, 0xFF, 0x0E, 0x28, 0x18, 0xE4, 0xC6, 0x16,
  0x15, 0x0E, 0x28, 0x18, 0xC1, 0x0E, 0x28, 0x18,
  0xBB, 0x05, 0xF9, 0x79, 0xBB, 0x0B, 0x1B, 0x6A,
  0xC6, 0x14, 0x50, 0x48, 0x55, 0x4E, 0x53, 0x59,
  0x20, 0x6D, 0x69, 0x63, 0x72, 0x6F, 0x63, 0x6F,
  0x6D, 0x70, 0x75, 0x74, 0x65, 0x72, 0xC6, 0x17,
  0x77, 0x69, 0x74, 0x68, 0x20, 0x53, 0x69, 0x67,
  0x6E, 0x65, 0x74, 0x69, 0x63, 0x73, 0x20, 0x32,
  0x36, 0x35, 0x30, 0x20, 0x63, 0x70, 0x75, 0xC6,
  0x11, 0x62, 0x79, 0x20, 0x46, 0x72, 0x61, 0x6E,
  0x6B, 0x20, 0x50, 0x68, 0x69, 0x6C, 0x69, 0x70,
  0x73, 0x65, 0xC6, 0x0F, 0x74, 0x68, 0x65, 0x20,
  0x4E, 0x65, 0x74, 0x68, 0x65, 0x72, 0x6C, 0x61,
  0x6E, 0x64, 0x73, 0xC6, 0x11, 0x77, 0x77, 0x77,
  0x2E, 0x70, 0x68, 0x69, 0x6C, 0x69, 0x70, 0x73,
  0x65, 0x2E, 0x69, 0x6E, 0x66, 0x6F, 0x00, 0x00,
  0x08, 0x1B, 0xCC, 0x0E, 0xE8, 0x08, 0x17, 0xCC,
  0x0E, 0xE9, 0x17, 0xD4, 0x7F, 0x1F, 0x05, 0xFA,
  0x05, 0x02, 0x0D, 0x45, 0xA2, 0xCD, 0x6E, 0xE8,
  0x59, 0x78, 0x17, 0xC4, 0x02, 0x08, 0x8B, 0x00
};

static const char *str1 = "PHUNSY microcomputer (Signetics 2650) emulator";
static const char *str2 = "Frank Philipse 2010-12-01 and "
                          "Fred N. van Kempen 2023-05-10";


/***************************************
** 2650 cpu definitions & variables
***************************************/

// addressing format
#define F_N  0  // no addressing
#define F_Z  1  // register addressing
#define F_I  2  // immediate
#define F_R  3  // relative
#define F_A  4  // absolute non branch
#define F_B  5  // absolute branch
#define F_C  6  // absolute program status
#define F_E  7  // miscellaneaus one byte instructions
#define F_EI 8  // miscellaneaus immediate
#define F_ER 9  // zero branch relative
#define F_EB 10 // absolute branch index R3

// psl bits
#define CY  (1<<0)
#define COM (1<<1)
#define OVF (1<<2)
#define WC  (1<<3)
#define RS  (1<<4)
#define IDC (1<<5)
#define CC0 (1<<6)
#define CC1 (1<<7)

// psu bits
#define SPMASK 0x07
// the rest is not interresting as there is no hardware. All 8 bits can be set or cleared by software (lpsu instruction) even the sense bit. The II bit is cleared by RETE.
#define UF1   (1<<3)		// only in 2650B
#define UF2   (1<<4)		// only in 2650B
#define II    (1<<5)
#define FLAG  (1<<6)		// used as audio cassette data output
#define SENSE (1<<7)		// used as audio cassette data input

static const struct instructions
{
  char mnem[8];
  int bytes;
  int format;
  int cycles;
} instrdata[256] =
{
//0x0x
  { "INVALID", 1, 0   , 0	},
  { "LODZ R1", 1, F_Z , 2	},
  { "LODZ R2", 1, F_Z , 2	},
  { "LODZ R3", 1, F_Z , 2	},
  { "LODI R0", 2, F_I , 2	},
  { "LODI R1", 2, F_I , 2	},
  { "LODI R2", 2, F_I , 2	},
  { "LODI R3", 2, F_I , 2	},
  { "LODR R0", 2, F_R , 3	},
  { "LODR R1", 2, F_R , 3	},
  { "LODR R2", 2, F_R ,	3	},
  { "LODR R3", 2, F_R , 3	},
  { "LODA R0", 3, F_A , 4	},
  { "LODA R1", 3, F_A , 4	},
  { "LODA R2", 3, F_A , 4	},
  { "LODA R3", 3, F_A , 4	},
// 0x1x
  { "LDPL   ", 3, F_C , 4	},	// 2650B only
  { "STPL   ", 3, F_C , 4	},	// 2650B only
  { "SPSU   ", 1, F_E , 2	},
  { "SPSL   ", 1, F_E , 2	},
  { "RETC  Z", 1, F_Z , 3	},
  { "RETC  P", 1, F_Z , 3	},
  { "RETC  N", 1, F_Z , 3	},
  { "RETC UN", 1, F_Z , 3	},
  { "BCTR  Z", 2, F_R , 3	},
  { "BCTR  P", 2, F_R , 3	},
  { "BCTR  N", 2, F_R , 3	},
  { "BCTR UN", 2, F_R , 3	},
  { "BCTA  Z", 3, F_B , 3	},
  { "BCTA  P", 3, F_B , 3	},
  { "BCTA  N", 3, F_B , 3	},
  { "BCTA UN", 3, F_B , 3	},
// 0x2x
  { "EORZ R0", 1, F_Z , 2	},
  { "EORZ R1", 1, F_Z , 2	},
  { "EORZ R2", 1, F_Z , 2	},
  { "EORZ R3", 1, F_Z , 2	},
  { "EORI R0", 2, F_I , 2	},
  { "EORI R1", 2, F_I , 2	},
  { "EORI R2", 2, F_I , 2	},
  { "EORI R3", 2, F_I , 2	},
  { "EORR R0", 2, F_R , 3	},
  { "EORR R1", 2, F_R , 3	},
  { "EORR R2", 2, F_R , 3	},
  { "EORR R3", 2, F_R , 3	},
  { "EORA R0", 3, F_A , 4	},
  { "EORA R1", 3, F_A , 4	},
  { "EORA R2", 3, F_A , 4	},
  { "EORA R3", 3, F_A , 4	},
// 0x3x
  { "REDC R0", 1, F_Z , 2	},
  { "REDC R1", 1, F_Z , 2	},
  { "REDC R2", 1, F_Z , 2	},
  { "REDC R3", 1, F_Z , 2	},
  { "RETE  Z", 1, F_Z , 3	},
  { "RETE  P", 1, F_Z , 3	},
  { "RETE  N", 1, F_Z , 3	},
  { "RETE UN", 1, F_Z , 3	},
  { "BSTR  Z", 2, F_R , 3	},
  { "BSTR  P", 2, F_R , 3	},
  { "BSTR  N", 2, F_R , 3	},
  { "BSTR UN", 2, F_R , 3	},
  { "BSTA  Z", 3, F_B , 3	},
  { "BSTA  P", 3, F_B , 3	},
  { "BSTA  N", 3, F_B , 3	},
  { "BSTA UN", 3, F_B , 3	},
// 0x4x
  { "HALT   ", 1, F_E , 1	},
  { "ANDZ R1", 1, F_Z , 2	},
  { "ANDZ R2", 1, F_Z , 2	},
  { "ANDZ R3", 1, F_Z , 2	},
  { "ANDI R0", 2, F_I , 2	},
  { "ANDI R1", 2, F_I , 2	},
  { "ANDI R2", 2, F_I , 2	},
  { "ANDI R3", 2, F_I , 2	},
  { "ANDR R0", 2, F_R , 3	},
  { "ANDR R1", 2, F_R , 3	},
  { "ANDR R2", 2, F_R , 3	},
  { "ANDR R3", 2, F_R , 3	},
  { "ANDA R0", 3, F_A , 4	},
  { "ANDA R1", 3, F_A , 4	},
  { "ANDA R2", 3, F_A , 4	},
  { "ANDA R3", 3, F_A , 4	},
// 0x5x
  { "RRR  R0", 1, F_Z , 2	},
  { "RRR  R1", 1, F_Z , 2	},
  { "RRR  R2", 1, F_Z , 2	},
  { "RRR  R3", 1, F_Z , 2	},
  { "REDE R0", 2, F_I , 3	},
  { "REDE R1", 2, F_I , 3	},
  { "REDE R2", 2, F_I , 3	},
  { "REDE R3", 2, F_I , 3	},
  { "BRNR R0", 2, F_R , 3	},
  { "BRNR R1", 2, F_R , 3	},
  { "BRNR R2", 2, F_R , 3	},
  { "BRNR R3", 2, F_R , 3	},
  { "BRNA R0", 3, F_B , 3	},
  { "BRNA R1", 3, F_B , 3	},
  { "BRNA R2", 3, F_B , 3	},
  { "BRNA R3", 3, F_B , 3	},
// 0x6
  { "IORZ R0", 1, F_Z , 2	},
  { "IORZ R1", 1, F_Z , 2	},
  { "IORZ R2", 1, F_Z , 2	},
  { "IORZ R3", 1, F_Z , 2	},
  { "IORI R0", 2, F_I , 2	},
  { "IORI R1", 2, F_I , 2	},
  { "IORI R2", 2, F_I , 2	},
  { "IORI R3", 2, F_I , 2	},
  { "IORR R0", 2, F_R , 3	},
  { "IORR R1", 2, F_R , 3	},
  { "IORR R2", 2, F_R , 3	},
  { "IORR R3", 2, F_R , 3	},
  { "IORA R0", 3, F_A , 4	},
  { "IORA R1", 3, F_A , 4	},
  { "IORA R2", 3, F_A , 4	},
  { "IORA R3", 3, F_A , 4	},
// 0x7x
  { "REDD R0", 1, F_Z , 2	},
  { "REDD R1", 1, F_Z , 2	},
  { "REDD R2", 1, F_Z , 2	},
  { "REDD R3", 1, F_Z , 2	},
  { "CPSU   ", 2, F_EI, 3	},
  { "CPSL   ", 2, F_EI, 3	},
  { "PPSU   ", 2, F_EI, 3	},
  { "PPSL   ", 2, F_EI, 3	},
  { "BSNR  Z", 2, F_R , 3	},
  { "BSNR  P", 2, F_R , 3	},
  { "BSNR  N", 2, F_R , 3	},
  { "BSNR UN", 2, F_R , 3	},
  { "BSNA  Z", 3, F_B , 3	},
  { "BSNA  P", 3, F_B , 3	},
  { "BSNA  N", 3, F_B , 3	},
  { "BSNA UN", 3, F_B , 3	},
// 0x8x
  { "ADDZ R0", 1, F_Z , 2	},
  { "ADDZ R1", 1, F_Z , 2	},
  { "ADDZ R2", 1, F_Z , 2	},
  { "ADDZ R3", 1, F_Z , 2	},
  { "ADDI R0", 2, F_I , 2	},
  { "ADDI R1", 2, F_I , 2	},
  { "ADDI R2", 2, F_I , 2	},
  { "ADDI R3", 2, F_I , 2	},
  { "ADDR R0", 2, F_R , 3	},
  { "ADDR R1", 2, F_R , 3	},
  { "ADDR R2", 2, F_R , 3	},
  { "ADDR R3", 2, F_R , 3	},
  { "ADDA R0", 3, F_A , 4	},
  { "ADDA R1", 3, F_A , 4	},
  { "ADDA R2", 3, F_A , 4	},
  { "ADDA R3", 3, F_A , 4	},
// 0x9x
  { "INVALID", 1, 0   , 0	},
  { "INVALID", 1, 0   , 0	},
  { "LPSU   ", 1, F_E , 2	},
  { "LPSL   ", 1, F_E , 2	},
  { "DAR  R0", 1, F_Z , 3	},	// cc is set to a meaningless value
  { "DAR  R1", 1, F_Z , 3	},
  { "DAR  R2", 1, F_Z , 3	},
  { "DAR  R3", 1, F_Z , 3	},
  { "BCFR  Z", 2, F_R , 3	},
  { "BCFR  P", 2, F_R , 3	},
  { "BCFR  N", 2, F_R , 3	},
  { "ZBRR   ", 2, F_ER, 3	},
  { "BCFA  Z", 3, F_B , 3	},
  { "BCFA  P", 3, F_B , 3	},
  { "BCFA  N", 3, F_B , 3	},
  { "BXA  R3", 3, F_EB, 3	},	// index register is R3 only for BXA
// 0xax
  { "SUBZ R0", 1, F_Z , 2	},
  { "SUBZ R1", 1, F_Z , 2	},
  { "SUBZ R2", 1, F_Z , 2	},
  { "SUBZ R3", 1, F_Z , 2	},
  { "SUBI R0", 2, F_I , 2	},
  { "SUBI R1", 2, F_I , 2	},
  { "SUBI R2", 2, F_I , 2	},
  { "SUBI R3", 2, F_I , 2	},
  { "SUBR R0", 2, F_R , 3	},
  { "SUBR R1", 2, F_R , 3	},
  { "SUBR R2", 2, F_R , 3	},
  { "SUBR R3", 2, F_R , 3	},
  { "SUBA R0", 3, F_A , 4	},
  { "SUBA R1", 3, F_A , 4	},
  { "SUBA R2", 3, F_A , 4	},
  { "SUBA R3", 3, F_A , 4	},
// 0xbx
  { "WRTC R0", 1, F_Z , 2	},
  { "WRTC R1", 1, F_Z , 2	},
  { "WRTC R2", 1, F_Z , 2	},
  { "WRTC R3", 1, F_Z , 2	},
  { "TPSU   ", 2, F_EI, 2	},
  { "TPSL   ", 2, F_EI, 2	},
  { "INVALID", 1, 0   , 0	},
  { "INVALID", 1, 0   , 0	},
  { "BSFR  Z", 2, F_R , 3	},
  { "BSFR  P", 2, F_R , 3	},
  { "BSFR  N", 2, F_R , 3	},
  { "ZBSR   ", 2, F_ER, 3	},
  { "BSFA  Z", 3, F_B , 3	},
  { "BSFA  P", 3, F_B , 3	},
  { "BSFA  N", 3, F_B , 3	},
  { "BSXA R3", 3, F_EB, 3	},	// index register is R3 only for BXA
// 0xcx
  { "NOP    ", 1, F_Z , 1	},
  { "STRZ R1", 1, F_Z , 2	},
  { "STRZ R2", 1, F_Z , 2	},
  { "STRZ R3", 1, F_Z , 2	},
  { "INVALID", 1, 0   , 0	},
  { "INVALID", 1, 0   , 0	},
  { "INVALID", 1, 0   , 0	},
  { "INVALID", 1, 0   , 0	},
  { "STRR R0", 2, F_R , 3	},
  { "STRR R1", 2, F_R , 3	},
  { "STRR R2", 2, F_R , 3	},
  { "STRR R3", 2, F_R , 3	},
  { "STRA R0", 3, F_A , 4	},
  { "STRA R1", 3, F_A , 4	},
  { "STRA R2", 3, F_A , 4	},
  { "STRA R3", 3, F_A , 4	},
// 0xdx
  { "RRL  R0", 1, F_Z , 2	},
  { "RRL  R1", 1, F_Z , 2	},
  { "RRL  R2", 1, F_Z , 2	},
  { "RRL  R3", 1, F_Z , 2	},
  { "WRTE R0", 2, F_I , 3	},
  { "WRTE R1", 2, F_I , 3	},
  { "WRTE R2", 2, F_I , 3	},
  { "WRTE R3", 2, F_I , 3	},
  { "BIRR R0", 2, F_R , 3	},
  { "BIRR R1", 2, F_R , 3	},
  { "BIRR R2", 2, F_R , 3	},
  { "BIRR R3", 2, F_R , 3	},
  { "BIRA R0", 3, F_B , 3	},
  { "BIRA R1", 3, F_B , 3	},
  { "BIRA R2", 3, F_B , 3	},
  { "BIRA R3", 3, F_B , 3	},
// 0xex
  { "COMZ R0", 1, F_Z , 2	},
  { "COMZ R1", 1, F_Z , 2	},
  { "COMZ R2", 1, F_Z , 2	},
  { "COMZ R3", 1, F_Z , 2	},
  { "COMI R0", 2, F_I , 2	},
  { "COMI R1", 2, F_I , 2	},
  { "COMI R2", 2, F_I , 2	},
  { "COMI R3", 2, F_I , 2	},
  { "COMR R0", 2, F_R , 3	},
  { "COMR R1", 2, F_R , 3	},
  { "COMR R2", 2, F_R , 3	},
  { "COMR R3", 2, F_R , 3	},
  { "COMA R0", 3, F_A , 4	},
  { "COMA R1", 3, F_A , 4	},
  { "COMA R2", 3, F_A , 4	},
  { "COMA R3", 3, F_A , 4	},
// oxfx
  { "WRTD R0", 1, F_Z , 2	},
  { "WRTD R1", 1, F_Z , 2	},
  { "WRTD R2", 1, F_Z , 2	},
  { "WRTD R3", 1, F_Z , 2	},
  { "TMI  R0", 2, F_I , 3	},
  { "TMI  R1", 2, F_I , 3	},
  { "TMI  R2", 2, F_I , 3	},
  { "TMI  R3", 2, F_I , 3	},
  { "BDRR R0", 2, F_R , 3	},
  { "BDRR R1", 2, F_R , 3	},
  { "BDRR R2", 2, F_R , 3	},
  { "BDRR R3", 2, F_R , 3	},
  { "BDRA R0", 3, F_B , 3	},
  { "BDRA R1", 3, F_B , 3	},
  { "BDRA R2", 3, F_B , 3	},
  { "BDRA R3", 3, F_B , 3	}
};

struct regset2650
{
  uint8_t r[7];		// R0 R1 R2 R3 R1' R2' R3'
  uint8_t psl;		// program status lower
  uint8_t psu;		// program status upper
  uint16_t ap;		// address pointer
  uint16_t as[8];	// address stack
  uint16_t na;		// next address (not part of 2650 regsiter set)
  uint16_t ea;		// effective address (not part of 2650 regsiter set)
} registers;

#define HISTORY_SIZE 10
struct regset2650 history[HISTORY_SIZE];
int historyindex;

int cpucycles;

/***************************************
** PHUNSY definitions & variables
***************************************/

/*
memory
0000-07FF monitor
0800-0FFF scratch ram (for monitor partly)
1000-17FF screen 64x32 characters
1800-1FFF U banks
2000-3FFF application memory
4000-7FFF Q banks
*/

#define PHSCRADR 0x1000             // phunsy screen address

uint8_t * memory;			// This is the entire 2650 memory 0000-7FFF
int pagelowptr;				// U monitor command - bank selected with Control port - bit 4-7 (WRTC instruction)
uint8_t * pagelow;		// 1800-1FFF low memory banked, this contains 16 banks of 2k bytes
int pagehighptr;				// Q monitor command - bank selected with Control port - bit 0-3 (WRTC instruction)
uint8_t * pagehigh;           // 4000-7FFF high memory banked, this contains 16 banks of 16 k bytes
uint8_t * extio;              // extended i/o

uint8_t portc_in;             // several input control bits
uint8_t portc_out;            // bank switching
uint8_t portd_in;             // keyboard input
uint8_t portd_out;            // several output control bits

uint8_t ext126_outold;        // to track changes for timer interrupt handling

// port d outputs
#define TTY_OUT      (1<<0)         // serial data out to tty (300 or 110 bps)
#define SPEAKER      (1<<1)         // output to a small speaker
#define KEYB_ACK     (1<<2)         // Clears the keyboard flip-flop, portd bit 8 is set to '1' indicating no key
#define MDCR_TST     (1<<3)         // Test CIP ??? Clears the CIP flip-flop
#define MDCR_REV     (1<<4)         // Reverse tape
#define MDCR_FWD     (1<<5)         // Forward tape
#define MDCR_WCD     (1<<6)         // Write command
#define MDCR_WDA     (1<<7)         // Write data

// port c inputs
#define TTY_IN       (1<<0)         // serial data in from tty (300 or 110 bps)
#define PAR_TTYN     (1<<1)         // if '1' keyboard and screen else tty in/out for user interface
#define BPS_300      (1<<2)         // if '1' 300 bps else 110 bps tty
#define MDCR_WEN     (1<<3)         // Write enable
#define MDCR_CIP     (1<<4)         // Cassette in position
#define MDCR_BET     (1<<5)         // Begin/End of Tape
#define MDCR_RDA     (1<<6)         // Read data
#define MDCR_RDC     (1<<7)         // Read clock

/***************************************
** mdcr definitions & variables
***************************************/
#define MDCR_IO 0x1f                // These are the bits in the control ports that control the mdcr

#define MDCR_TAPEEND              30853345   // 92.56 seconds tape length.
#define MDCR_OFFSET               333333     // 1 second
#define MDCR_BLOCK                233333     // 700 ms total data block time
#define MDCR_BLOCKDATLEN          115111     // length of data block in cpu cycles
#define MDCR_DATA                 25000      // 75 ms  offset to data in each block
#define MDCR_BLOCKNUMLEN          1333       // length of blocknumber in cpu cycles
#define MDCR_BIT                  56         // 6000 bps

#define MDCRSTATE_IDLE            0
#define MDCRSTATE_ATSTART         1
#define MDCRSTATE_ATEND           2
#define MDCRSTATE_FWD             3
#define MDCRSTATE_REV             4

#define MDCRFWDSTATE_IDLE         0
#define MDCRFWDSTATE_NUMREAD_A    1
#define MDCRFWDSTATE_NUMREAD_B    2
#define MDCRFWDSTATE_BLOCKREAD_A  3
#define MDCRFWDSTATE_BLOCKREAD_B  4
#define MDCRFWDSTATE_WRITE        5
#define MDCRFWDSTATE_WRITE_ACTIVE 6
#define MDCRFWDSTATE_INIT         7

#define MDCRREVSTATE_LGAP         0
#define MDCRREVSTATE_DATA         1
#define MDCRREVSTATE_SGAP         2
#define MDCRREVSTATE_BNUM         3

uint8_t * cassette;           // memory representing a cassette (128 blocks of 256 bytes)
uint8_t * cassettes;
int selectedcassette;
int cassettechanged[4];

int mdcrstate;
int mdcrstateold;
int mdcrfwdstate;
int mdcrpos;
int mdcrblock;
int mdcrdatapos;
int mdcrbitcounter;
uint8_t mdcrchecksum;
int mdcrsecond;
int mdcrwritebittime;
int mdcrportdoutold;
uint8_t mdcrwritebyte;
int mdcrblockpos;                            // position within a block
int mdcrrevstate;

/***************************************
** Emulator definitions & variables
***************************************/

int options;
#define PRINT             (1<<0)
#define STOP_STACK_OVF    (1<<1)
#define INFILE            (1<<2)
#define TRACE             (1<<3)
#define CASSETTE_CHANGED  (1<<4)
#define LOGFILE           (1<<5)
#define FULLSCREEN        (1<<6)
#define GREENSCRN         (1<<7)


/***************************************
** 2650 Emulator
***************************************/

// perform add and set ICD, CY and OVF accordingly
// this seems to work correctly
static uint8_t add(uint8_t byte1, uint8_t byte2)
{
  uint16_t result, result2;

  result=0;
  if(((registers.psl)&(WC|CY))==(WC|CY))
    result=1;
  result2=result;
  result+=byte1+byte2;
  result2+=(byte1&15)+(byte2&15);

  registers.psl&=~IDC;
  if((result2&0x0010)!=0)
    registers.psl|=IDC;

  registers.psl&=~CY;
    if((result&0x0100)!=0)
  registers.psl|=CY;

  registers.psl&=~OVF;
  if((byte1&0x80)==(byte2&0x80))
    if((byte1&0x80)!=(result&0x0080))
      registers.psl|=OVF;

  return((uint8_t)result);
}

// perform subtract and set ICD, CY and OVF accordingly
// note: borrow is stored inverted in program status. IDC too?
static uint8_t sub(uint8_t byte1, uint8_t byte2)
{
  uint16_t result, result2;

  result=0;
  if(((registers.psl)&(WC|CY))==WC)
    result=1;
  result2=result;
  result=byte1-byte2-result;
  result2=(byte1&15)-(byte2&15)-result2;

  registers.psl&=~IDC;
  if((result2&0x0010)==0)
    registers.psl|=IDC;

  registers.psl&=~CY;
    if((result&0x0100)==0)
  registers.psl|=CY;

  registers.psl&=~OVF;
  if((byte1&0x80)==(byte2&0x80))
    if((byte1&0x80)!=(result&0x0080))
      registers.psl|=OVF;

  return((uint8_t)result);
}

// perform compare
// returns positive byte if byte1 > byte2
// returns negative byte if byte1 < byte2
// returns zero byte if byte1 = byte2
// COM in psl determines compare mode:
// 0 means signed compare, 1 means unsigned compare
static uint8_t com(uint8_t byte1, uint8_t byte2)
{
  uint16_t result;

  result=0;
  if((registers.psl&COM)!=0)                // unsigned compare
  {
    if(((byte1&0x80)!=0)&&((byte2&0x80)==0))
      result=0x0001;
    if(((byte1&0x80)==0)&&((byte2&0x80)!=0))
      result=0xffff;
  }
  else                                      // signed compare
  {
    if(((byte1&0x80)!=0)&&((byte2&0x80)==0))
      result=0xffff;
    if(((byte1&0x80)==0)&&((byte2&0x80)!=0))
      result=0x0001;
  }
  if(result==0)
    result=byte1-byte2;

  return((uint8_t)result);
}

// perform right rotate and set ICD, CY and OVF accordingly
static uint8_t rrr(uint8_t byte)
{
  uint16_t result;

  result=byte;
  if((registers.psl&WC)!=0)
  {
    if((registers.psl&CY)!=0)
      result|=0x0100;
    registers.psl&=~CY;
    if((result&0x0001)!=0)
      registers.psl|=CY;
    registers.psl&=~IDC;
    if((result&0x0040)!=0)
      registers.psl|=IDC;
  }
  else
  {
    if((result&0x0001)!=0)
      result|=0x0100;
  }
  result=result>>1;
  return((uint8_t)result);
}

// perform left rotate and set ICD, CY and OVF accordingly
static uint8_t rrl(uint8_t byte)
{
  uint16_t result;

  result=byte<<1;
  if((registers.psl&WC)!=0)
  {
    if((registers.psl&CY)!=0)
      result|=0x0001;
    registers.psl&=~CY;
    if((result&0x0100)!=0)
      registers.psl|=CY;
    registers.psl&=~IDC;
    if((result&0x0020)!=0)
      registers.psl|=IDC;
  }
  else
  {
    if((result&0x0100)!=0)
      result|=0x0001;
  }
  return((uint8_t)result);
}

// set cc according to byte
static void setcc(uint8_t byte)
{
    registers.psl&=~(CC1|CC0);
    if((byte&0x80)!=0)
      registers.psl|=CC1;
    else
      if((byte&0x7f)!=0)
        registers.psl|=CC0;
}

// push address on stack
static void push(uint16_t address)
{
  registers.as[registers.psu&0x07]=address; // store address
  registers.psu=((registers.psu+1)&0x07)|(registers.psu&0xf8); // increment stack pointer and leave other bits
}

// pull address from stack
static uint16_t pull(void)
{
  registers.psu=((registers.psu-1)&0x07)|(registers.psu&0xf8); // increment stack pointer and leave other bits
  return(registers.as[registers.psu&0x07]);
}

static void interrupt(void)
{
  push(registers.na);
  registers.na=0x001D; // this is the same as 'ZBSR 001d'
}

// cpu function return codes
#define NORMAL      (0<<16)  // just registers have changed
#define INV_OPCODE  (1<<16)  // invalid opcode
#define HALT        (2<<16)  // HALT instruction encounted
#define MEM_WRITE   (3<<16)  // a location in memory has been altered
#define MEM_READ    (4<<16)  // a location in memory has been read
#define PC_WRITE    (5<<16)  // A write to port-c
#define PC_READ     (6<<16)  // A read from port-c
#define PD_WRITE    (7<<16)  // A write to port-d
#define PD_READ     (8<<16)  // A read from port-d
#define EXT_WRITE   (9<<16)  // A write to port-d
#define EXT_READ   (10<<16)  // A read from port-d

// this executes one instruction
static int cpu(void)
{
  int ret; // return code
  uint8_t instrbyte1 = 0;
  uint8_t instrbyte2 = 0;
  uint8_t instrbyte3 = 0;
  uint16_t address;
  uint16_t effaddr = 0;  // effective address
  uint16_t tmpaddr;
  int numbytes;
  int regnum; // 0..3 and 4..6
  int result;
  int ind;    // 2 extra cycles for indirect addressing

  registers.ap=registers.na;
  ind=0;
  ret=(NORMAL|registers.ap);
  address = registers.ap;
  instrbyte1 = memory[address];
  numbytes=instrdata[instrbyte1].bytes;
  cpucycles=instrdata[instrbyte1].cycles;
  address=((address+1)&0x1fff)|(address&0x6000);
  if(numbytes>1)
  {
    instrbyte2 = memory[address];
    address=((address+1)&0x1fff)|(address&0x6000);
    if(numbytes>2)
    {
      instrbyte3 = memory[address];
      address=((address+1)&0x1fff)|(address&0x6000);
    }
  }

  regnum=instrbyte1&0x03; // in many cases
  if(regnum!=0)
    if((registers.psl&RS)!=0)
      regnum+=3;

  switch(instrdata[instrbyte1].format)
  {
    case F_R  : // relative
      if((instrbyte2&0x40)==0)
        tmpaddr=((address+(instrbyte2&0x3f))&0x1fff)|(address&0x6000);
      else
        tmpaddr=(address-((~instrbyte2&0x3f)+1))|(address&0x6000);
      if((instrbyte2&0x80)==0)
        effaddr=tmpaddr;
      else // indirect
      {
        effaddr=((memory[tmpaddr]<<8) | memory[((tmpaddr+1)&0x1fff)|(tmpaddr&0x6000)]) & 0x7fff;
        ind=2;
      }
      break;
    case F_A  : // absolute, non branch instructions
      tmpaddr= (((instrbyte2<<8) | instrbyte3) & 0x1fff) | (address&0x6000);
      if((instrbyte2&0x80)==0)
        effaddr=tmpaddr;
      else // indirect
      {
        effaddr=((memory[tmpaddr]<<8) | memory[((tmpaddr+1)&0x1fff)|(tmpaddr&0x6000)]) & 0x7fff;
        ind=2;
      }
      if((instrbyte2&0x60)!=0)  // indexed
      {
        if((instrbyte2&0x60)==0x20)
          registers.r[regnum]++;
        if((instrbyte2&0x60)==0x40)
          registers.r[regnum]--;
        effaddr+=registers.r[regnum];
        effaddr&=0x7fff;
        regnum=0;
      }
      break;
    case F_B  : // absolute branch
    case F_C  : // absolute program status
    case F_EB : // absolute indexed R3
      tmpaddr= ((instrbyte2<<8) | instrbyte3) & 0x7fff;
      if((instrbyte2&0x80)==0)
        effaddr=tmpaddr;
      else // indirect
      {
        effaddr=((memory[tmpaddr]<<8) | memory[((tmpaddr+1)&0x1fff)|(tmpaddr&0x6000)]) & 0x7fff;
        ind=2;
      }
      if(instrdata[instrbyte1].format==F_EB)
        effaddr=effaddr+(registers.r[regnum]&0x7fff);
      break;
    case F_ER : // zero branch
      if((instrbyte2&0x40)==0)
        tmpaddr=instrbyte2&0x3f;
      else
        tmpaddr=(0x2000-((~instrbyte2&0x3f)+1));
      if((instrbyte2&0x80)==0)
        effaddr=tmpaddr;
      else // indirect
      {
        effaddr=((memory[tmpaddr]<<8) | memory[((tmpaddr+1)&0x1fff)|(tmpaddr&0x6000)]) & 0x7fff;
        ind=2;
      }
      break;
  }


  switch(instrbyte1)
  {
// LOD instructuins
    case 0x01 : // lodz r1
    case 0x02 : // lodz r2
    case 0x03 : // lodz r3
      result=registers.r[regnum];
      registers.r[0]=result;
      setcc(result);
      break;
    case 0x04 : // lodi r0
    case 0x05 : // lodi r1
    case 0x06 : // lodi r2
    case 0x07 : // lodi r3
      result=instrbyte2;
      registers.r[regnum]=result;
      setcc(result);
      break;
    case 0x08 : // lodr r0
    case 0x09 : // lodr r1
    case 0x0a : // lodr r2
    case 0x0b : // lodr r3
    case 0x0c : // loda r0
    case 0x0d : // loda r1
    case 0x0e : // loda r2
    case 0x0f : // loda r3
      result=memory[effaddr];
      registers.r[regnum]=result;
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// EOR instructions
    case 0x20 : // eorz r0
    case 0x21 : // eorz r1
    case 0x22 : // eorz r2
    case 0x23 : // eorz r3
      result=(registers.r[regnum]&~registers.r[0])|(~registers.r[regnum]&registers.r[0]);
      registers.r[0]=result;
      setcc(result);
      break;
    case 0x24 : // eori r0
    case 0x25 : // eori r1
    case 0x26 : // eori r2
    case 0x27 : // eori r3
      result=(registers.r[regnum]&~instrbyte2)|(~registers.r[regnum]&instrbyte2);
      registers.r[regnum]=result;
      setcc(result);
      break;
    case 0x28 : // eorr r0
    case 0x29 : // eorr r1
    case 0x2a : // eorr r2
    case 0x2b : // eorr r3
    case 0x2c : // eora r0
    case 0x2d : // eora r1
    case 0x2e : // eora r2
    case 0x2f : // eora r3
      result=(registers.r[regnum]&~memory[effaddr])|(~registers.r[regnum]&memory[effaddr]);
      registers.r[regnum]=result;
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// AND instructions
    case 0x41 : // andz r1
    case 0x42 : // andz r2
    case 0x43 : // andz r3
      result=registers.r[regnum]&registers.r[0];
      registers.r[0]=result;
      setcc(result);
      break;
    case 0x44 : // andi r0
    case 0x45 : // andi r1
    case 0x46 : // andi r2
    case 0x47 : // andi r3
      result=registers.r[regnum]&instrbyte2;
      registers.r[regnum]=result;
      setcc(result);
      break;
    case 0x48 : // andr r0
    case 0x49 : // andr r1
    case 0x4a : // andr r2
    case 0x4b : // andr r3
    case 0x4c : // anda r0
    case 0x4d : // anda r1
    case 0x4e : // anda r2
    case 0x4f : // anda r3
      result=registers.r[regnum]&memory[effaddr];
      registers.r[regnum]=result;
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// IOR instructions
    case 0x60 : // iorz ro
    case 0x61 : // iorz r1
    case 0x62 : // iorz r2
    case 0x63 : // iorz r3
      result=registers.r[regnum]|registers.r[0];
      registers.r[0]=result;
      setcc(result);
      break;
    case 0x64 : // iori r0
    case 0x65 : // iori r1
    case 0x66 : // iori r2
    case 0x67 : // iori r3
      result=registers.r[regnum]|instrbyte2;
      registers.r[regnum]=result;
      setcc(result);
      break;
    case 0x68 : // iorr r0
    case 0x69 : // iorr r1
    case 0x6a : // iorr r2
    case 0x6b : // iorr r3
    case 0x6c : // iora r0
    case 0x6d : // iora r1
    case 0x6e : // iora r2
    case 0x6f : // iora r3
      result=registers.r[regnum]|memory[effaddr];
      registers.r[regnum]=result;
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// ADD instructions
    case 0x80 : // addz r0
    case 0x81 : // addz r1
    case 0x82 : // addz r2
    case 0x83 : // addz r3
      result=add(registers.r[0],registers.r[regnum]);
      registers.r[0]=result;
      setcc(result);
      break;
    case 0x84 : // addi r0
    case 0x85 : // addi r1
    case 0x86 : // addi r2
    case 0x87 : // addi r3
      result=add(registers.r[regnum],instrbyte2);
      registers.r[regnum]=result;
      setcc(result);
      break;
    case 0x88 : // addr r0
    case 0x89 : // addr r1
    case 0x8a : // addr r2
    case 0x8b : // addr r3
    case 0x8c : // adda r0
    case 0x8d : // adda r1
    case 0x8e : // adda r2
    case 0x8f : // adda r3
      result=add(registers.r[regnum],memory[effaddr]);
      registers.r[regnum]=result;
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// SUB instructions
    case 0xa0 : // subz r0
    case 0xa1 : // subz r1
    case 0xa2 : // subz r2
    case 0xa3 : // subz r3
      result=sub(registers.r[0],registers.r[regnum]);
      registers.r[0]=result;
      setcc(result);
      break;
    case 0xa4 : // subi r0
    case 0xa5 : // subi r1
    case 0xa6 : // subi r2
    case 0xa7 : // subi r3
      result=sub(registers.r[regnum],instrbyte2);
      registers.r[regnum]=result;
      setcc(result);
      break;
    case 0xa8 : // subr r0
    case 0xa9 : // subr r1
    case 0xaa : // subr r2
    case 0xab : // subr r3
    case 0xac : // suba r0
    case 0xad : // suba r1
    case 0xae : // suba r2
    case 0xaf : // suba r3
      result=sub(registers.r[regnum],memory[effaddr]);
      registers.r[regnum]=result;
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// STR instructions
    case 0xc1 : // strz r1
    case 0xc2 : // strz r2
    case 0xc3 : // strz r3
      registers.r[regnum]=registers.r[0];
      break;
    case 0xc8 : // strr r0
    case 0xc9 : // strr r1
    case 0xca : // strr r2
    case 0xcb : // strr r3
    case 0xcc : // stra r0
    case 0xcd : // stra r1
    case 0xce : // stra r2
    case 0xcf : // stra r3
      memory[effaddr]=registers.r[regnum];
      cpucycles+=ind;
      ret=MEM_WRITE|effaddr;
      break;
// COM instructions
    case 0xe0 : // comz r0
    case 0xe1 : // comz r1
    case 0xe2 : // comz r2
    case 0xe3 : // comz r3
      result=com(registers.r[0],registers.r[regnum]);
      setcc(result);
      break;
    case 0xe4 : // comi r0
    case 0xe5 : // comi r1
    case 0xe6 : // comi r2
    case 0xe7 : // comi r3
      result=com(registers.r[regnum],instrbyte2);
      setcc(result);
      break;
    case 0xe8 : // comr r0
    case 0xe9 : // comr r1
    case 0xea : // comr r2
    case 0xeb : // comr r3
    case 0xec : // coma r0
    case 0xed : // coma r1
    case 0xee : // coma r2
    case 0xef : // coma r3
      result=com(registers.r[regnum],memory[effaddr]);
      setcc(result);
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
// 0x18+
    case 0x18 : // bctr  z
    case 0x19 : // bctr  p
    case 0x1a : // bctr  n
    case 0x1c : // bcta  z
    case 0x1d : // bcta  p
    case 0x1e : // bcta  n
      if((instrbyte1&0x03) == ((registers.psl>>6)&0x03))
      {
        address=effaddr;
        cpucycles+=ind;
      }
      break;
    case 0x1b : // bctr un
    case 0x1f : // bcta un
      address=effaddr;
      cpucycles+=ind;
      break;
// 0x38+
    case 0x38 : // bstr  z
    case 0x39 : // bstr  p
    case 0x3a : // bstr  n
    case 0x3c : // bsta  z
    case 0x3d : // bsta  p
    case 0x3e : // bsta  n
      if((instrbyte1&0x03) == ((registers.psl>>6)&0x03))
      {
        push(address);
        address=effaddr;
        cpucycles+=ind;
      }
      break;
    case 0x3b : // bstr un
    case 0x3f : // bsta un
      push(address);
      address=effaddr;
      cpucycles+=ind;
      break;
// 0x58+
    case 0x58 : // brnr r0
    case 0x59 : // brnr r1
    case 0x5a : // brnr r2
    case 0x5b : // brnr r3
    case 0x5c : // brna r0
    case 0x5d : // brna r1
    case 0x5e : // brna r2
    case 0x5f : // brna r3
      if(registers.r[regnum]!=0)
      {
        address=effaddr;
        cpucycles+=ind;
      }
      break;
// 0x78+
    case 0x78 : // bsnr r0
    case 0x79 : // bsnr r1
    case 0x7a : // bsnr r2
    case 0x7b : // bsnr r3
    case 0x7c : // bsna r0
    case 0x7d : // bsna r1
    case 0x7e : // bsna r2
    case 0x7f : // bsna r3
      if(registers.r[regnum]!=0)
      {
        push(address);
        address=effaddr;
        cpucycles+=ind;
      }
      break;
// 0x98+
    case 0x98 : // bcfr  z
    case 0x99 : // bcfr  p
    case 0x9a : // bcfr  n
    case 0x9c : // bcfa  z
    case 0x9d : // bcfa  p
    case 0x9e : // bcfa  n
      if((instrbyte1&0x03) != ((registers.psl>>6)&0x03))
      {
        address=effaddr;
        cpucycles+=ind;
      }
      break;
    case 0x9b : // zbrr
    case 0x9f : // bxa
      address=effaddr;
      cpucycles+=ind;
      break;
// 0xb8+
    case 0xb8 : // bsfr  z
    case 0xb9 : // bsfr  p
    case 0xba : // bsfr  n
    case 0xbc : // bsfa  z
    case 0xbd : // bsfa  p
    case 0xbe : // bsfa  n
      if((instrbyte1&0x03) != ((registers.psl>>6)&0x03))
      {
        push(address);
        address=effaddr;
        cpucycles+=ind;
      }
      break;
    case 0xbb : // zbsr
    case 0xbf : // bsxa
      push(address);
      address=effaddr;
      cpucycles+=ind;
      break;
// 0xd8+
    case 0xd8 : // birr r0
    case 0xd9 : // birr r1
    case 0xda : // birr r2
    case 0xdb : // birr r3
    case 0xdc : // bira r0
    case 0xdd : // bira r1
    case 0xde : // bira r2
    case 0xdf : // bira r3
      registers.r[regnum]++;
      if(registers.r[regnum]!=0)
      {
        address=effaddr;
        cpucycles+=ind;
      }
      break;
// 0xf8+
    case 0xf8 : // bdrr r0
    case 0xf9 : // bdrr r1
    case 0xfa : // bdrr r2
    case 0xfb : // bdrr r3
    case 0xfc : // bdra r0
    case 0xfd : // bdra r1
    case 0xfe : // bdra r2
    case 0xff : // bdra r3
      registers.r[regnum]--;
      if(registers.r[regnum]!=0)
      {
        address=effaddr;
        cpucycles+=ind;
      }
      break;

// 0x10+
    case 0x10 : // ldpl // 2650B only
      registers.psl=memory[effaddr];
      cpucycles+=ind;
      ret=MEM_READ|effaddr;
      break;
    case 0x11 : // stpl // 2650B only
      memory[effaddr]=registers.psl;
      cpucycles+=ind;
      ret=MEM_WRITE|effaddr;
      break;
    case 0x12 : // spsu
      registers.r[0]=registers.psu;
      setcc(registers.r[0]);
      break;
    case 0x13 : // spsl
      registers.r[0]=registers.psl;
      setcc(registers.r[0]);
      break;

// 0x14+
    case 0x14 : // retc  z
    case 0x15 : // retc  p
    case 0x16 : // retc  n
      if((instrbyte1&0x03) == ((registers.psl>>6)&0x03))
        address=pull();
      break;
    case 0x17 : // retc un
      address=pull();
      break;

// 0x30+ // the C-port was used for several hardware functions e.g. the mdcr interface.
// bit 1 and 2 determine the interface
    case 0x30 : // redc r0
    case 0x31 : // redc r1
    case 0x32 : // redc r2
    case 0x33 : // redc r3
      result=portc_in;
      registers.r[regnum]=result;
      setcc(result);
      ret=PC_READ;
      break;

// 0x34+
    case 0x34 : // rete  z
    case 0x35 : // rete  p
    case 0x36 : // rete  n
      if((instrbyte1&0x03) == ((registers.psl>>6)&0x03))
      {
        registers.psu&=~II; // clear interrupt inhibit in the psu
        address=pull();
      }
      break;
    case 0x37 : // rete un
      registers.psu&=~II; // clear interrupt inhibit in the psu
      address=pull();
      break;

// 0x50+
    case 0x50 : // rrr r0
    case 0x51 : // rrr r1
    case 0x52 : // rrr r2
    case 0x53 : // rrr r3
      result=rrr(registers.r[regnum]);
      registers.r[regnum]=result;
      setcc(result);
      break;

// 0x54+
    case 0x54 : // rede r0
    case 0x55 : // rede r1
    case 0x56 : // rede r2
    case 0x57 : // rede r3
      if(instrbyte2<128)
        ret=instrbyte2;
      else
        ret=(instrbyte2-128)*(portc_out/16)+128;
      registers.r[regnum]=extio[ret];
      setcc(registers.r[regnum]);
      ret|=EXT_READ;
      break;

// 0x70+
    case 0x70 : // redd r0
    case 0x71 : // redd r1
    case 0x72 : // redd r2
    case 0x73 : // redd r3
      result=portd_in;
      registers.r[regnum]=result;
      setcc(result);
      ret=PD_READ;
      break;

// 0x74+
    case 0x74 : // cpsu
      registers.psu&=registers.psu&~instrbyte2;
      break;
    case 0x75 : // cpsl
      registers.psl&=registers.psl&~instrbyte2;
      break;
    case 0x76 : // ppsu
      registers.psu=registers.psu|instrbyte2;
      break;
    case 0x77 : // ppsl
      registers.psl=registers.psl|instrbyte2;
      break;

// 0x90+
    case 0x92 : // lpsu
      registers.psu=registers.r[0];
      break;
    case 0x93 : // lpsl
      registers.psl=registers.r[0];
      break;

// 0x90+
  case 0x94 : // dar r0
  case 0x95 : // dar r1
  case 0x96 : // dar r2
  case 0x97 : // dar r3
    result=registers.r[regnum];
    if((registers.psl&CY)==0)
      result=(result&0x0f)|((result+0xa0)&0xf0);
    if((registers.psl&IDC)==0)
      result=(result&0xf0)|((result+0x0a)&0x0f);
    registers.r[regnum]=result;
      break;

// 0xb0+
// this causes bank switching.
    case 0xb0 : // wrtc r0
    case 0xb1 : // wrtc r1
    case 0xb2 : // wrtc r2
    case 0xb3 : // wrtc r3
      portc_out=registers.r[regnum];
      ret=PC_WRITE;
      break;

    case 0xb4 : // tpsu
      result=registers.psu&instrbyte2;
      registers.psl&=~(CC0|CC1); // clear cc bits
      if(result!=instrbyte2)
        registers.psl|=0x80; // indicate not all selected bits are '1'
      break;
    case 0xb5 : // tpsl
      result=registers.psl&instrbyte2;
      registers.psl&=~(CC0|CC1); // clear cc bits
      if(result!=instrbyte2)
        registers.psl|=0x80; // indicate not all selected bits are '1'
      break;

// 0xd0+
    case 0xd0 : // rrl r0
    case 0xd1 : // rrl r1
    case 0xd2 : // rrl r2
    case 0xd3 : // rrl r3
      result=rrl(registers.r[regnum]);
      registers.r[regnum]=result;
      setcc(result);
      break;

// 0xb4+
    case 0xd4 : // wrte r0
    case 0xd5 : // wrte r1
    case 0xd6 : // wrte r2
    case 0xd7 : // wrte r3
      if(instrbyte2<128)
        ret=instrbyte2;
      else
        ret=(instrbyte2-128)*(portc_out/16)+128;
      extio[ret]=registers.r[regnum];
      ret|=EXT_WRITE;
      break;

// 0xf0+
    case 0xf0 : // wrtd r0
    case 0xf1 : // wrtd r1
    case 0xf2 : // wrtd r2
    case 0xf3 : // wrtd r3
      portd_out=registers.r[regnum];
      ret=PD_WRITE;
      break;

// 0xf4+
    case 0xf4 : // tmi r0
    case 0xf5 : // tmi r1
    case 0xf6 : // tmi r2
    case 0xf7 : // tmi r3
      result=registers.r[regnum]&instrbyte2;
      registers.psl&=~(CC0|CC1); // clear cc bits
      if(result!=instrbyte2)
        registers.psl|=0x80; // indicate not all selected bits are '1'
      break;

// misc instructions
    case 0x40 : // halt
      ret = HALT|registers.ap;
      break;
    case 0xc0 : // nop
      break;    // do nothing

    default : // invalid opcode
      ret=(INV_OPCODE|registers.ap); // invalid instruction
      break;
  }

  registers.ea=effaddr;
  registers.na=address;
  return(ret);
}


static void disassemble(void)
{
  uint8_t instrbyte1;
  uint8_t instrbyte2 = 0;
  uint8_t instrbyte3 = 0;
  uint16_t address;
  int numbytes;

  address = registers.ap;
  instrbyte1 = memory[address];
  numbytes=instrdata[instrbyte1].bytes;
  if(numbytes>1)
  {
    address=((address+1)&0x1fff)|(address&0x6000);
    instrbyte2 = memory[address];
    if(numbytes>2)
    {
      address=((address+1)&0x1fff)|(address&0x6000);
      instrbyte3 = memory[address];
    }
  }

  fprintf(stdout, "%04X %02X ", registers.ap, instrbyte1);
  if(numbytes>1)
    fprintf(stdout, "%02X ", instrbyte2);
  else
    fprintf(stdout, "   ");
  if(numbytes>2)
    fprintf(stdout, "%02X ", instrbyte3);
  else
    fprintf(stdout, "   ");
  fprintf(stdout, "%s ", instrdata[instrbyte1].mnem);
  switch(instrdata[instrbyte1].format)
  {
    case F_R :
    case F_A :
    case F_B :
    case F_C :
    case F_EB :
    case F_ER :
     fprintf(stdout, "%04X ", registers.ea);
     break;
    case F_I :
     fprintf(stdout, "%02X   ", instrbyte2);
     break;
    default :
//    fprintf(stdout, "     ", registers.ea);
     fprintf(stdout, "     ");
     break;
  }
  switch(instrdata[instrbyte1].format)
  {
    case F_R :
    case F_A :
    case F_B :
    case F_C :
    case F_EB :
    case F_ER :
      if((instrbyte2&0x80)!=0)
        fprintf(stdout, "* ");
      else
        fprintf(stdout, "  ");
      break;
    default :
      fprintf(stdout, "  ");
      break;
  }
  switch(instrdata[instrbyte1].format)
  {
    case F_A :
    case F_C :
      switch(instrbyte2&0x60)
      {
        case 0x20 :
          fprintf(stdout, "#+ ");
          break;
        case 0x40 :
          fprintf(stdout, "#- ");
          break;
        case 0x60 :
          fprintf(stdout, "#  ");
          break;
        default :
          fprintf(stdout, "   ");
          break;
      }
      break;
    default :
      fprintf(stdout, "   ");
      break;
  }
  fprintf(stdout, "%02X ", registers.r[0]);
  fprintf(stdout, "%02X ", registers.r[1]);
  fprintf(stdout, "%02X ", registers.r[2]);
  fprintf(stdout, "%02X ", registers.r[3]);
  fprintf(stdout, "%02X ", registers.r[4]);
  fprintf(stdout, "%02X ", registers.r[5]);
  fprintf(stdout, "%02X ", registers.r[6]);
  fprintf(stdout, "%02X ", registers.psu);
  fprintf(stdout, "%02X ", registers.psl);
  fprintf(stdout, "\n");
}


/***************************************
** phunsy hardware simulation
***************************************/

/*
MDCR info
total tape length is 1+128*0.7+1 = 93 s is 31,000,000 cpu cycles
the cassette length is 128 blocks of 700ms = 233,333 cpu cycles / block.
first block starts at 1s = 333,333 cpu cycles
block header to block is 75ms = 25,000 cpu cycles
1 bit time is 166.666 s = 55.555 cpu cycles
----
block number starts at n * 233333 cycles
data block   starts at n * 233333+25000 cycles

If both fwd and rev are asserted, the tape goes forward.
*/

void mdcr_exec(void)
{
  int i;

  switch(mdcrstate)
  {
    default :
    case MDCRSTATE_IDLE :
      mdcrsecond=0;           // for debug
      mdcrfwdstate=MDCRFWDSTATE_IDLE;
      if((portd_out&MDCR_FWD)!=0)
      {
        mdcrstate=MDCRSTATE_FWD;
        break;
      }
      if((portd_out&MDCR_REV)!=0)
      {
        mdcrstate=MDCRSTATE_REV;
        break;
      }
      break;
    case MDCRSTATE_FWD :
      if((portd_out&MDCR_FWD)==0) // tape stop
      {
        mdcrstate=MDCRSTATE_IDLE;
        break;
      }
      mdcrpos+=cpucycles;
      if(mdcrpos>=MDCR_TAPEEND)
      {
        mdcrpos=MDCR_TAPEEND;
        mdcrstate=MDCRSTATE_ATEND;
        break;
      }
      switch(mdcrfwdstate)
      {
        case MDCRFWDSTATE_INIT :
          mdcrsecond+=cpucycles;
          if(mdcrsecond>333333)
          {
#ifdef MDCRDEBUG
            fprintf(stdout, "cassette position %8d\n", mdcrpos);
#endif
            mdcrsecond=0;
          }
          if((portd_out&MDCR_WCD)==0) // end of write
            mdcrfwdstate=MDCRFWDSTATE_IDLE;
          break;
        case MDCRFWDSTATE_WRITE :
          if((portd_out&MDCR_WCD)==0) // end of write
          {
#ifdef MDCRDEBUG
            fprintf(stdout, "cassette position %8d block number %3d write negated\n", mdcrpos, mdcrblock);
#endif
            mdcrfwdstate=MDCRFWDSTATE_IDLE;
          }
          else if(mdcrpos<100)
          {
            mdcrfwdstate=MDCRFWDSTATE_INIT;
            for(i=0; i<32768; i++)
              cassette[i]=0;
          }
          else if((portd_out&MDCR_WDA)!=0) // detected 1st bit, 1st bit is always inverse of 0
          {
            mdcrfwdstate=MDCRFWDSTATE_WRITE_ACTIVE;
            mdcrwritebittime=0;
            mdcrbitcounter=1;
            mdcrwritebyte=0;
          }
          break;
        case MDCRFWDSTATE_WRITE_ACTIVE :
          if((portd_out&MDCR_WCD)==0) // end of write
          {
#ifdef MDCRDEBUG
            fprintf(stdout, "cassette position %8d block number %3d write negated from active\n", mdcrpos, mdcrblock);
#endif
            mdcrfwdstate=MDCRFWDSTATE_IDLE;
          }
          mdcrwritebittime+=cpucycles;
          if(mdcrwritebittime<(MDCR_BIT*3/4))
          {
            mdcrportdoutold=portd_out;
            break;
          }
          else
          {
            if(((mdcrportdoutold^portd_out)&MDCR_WDA)!=0)    // detected new bit
            {
              mdcrwritebyte>>=1;
              if((portd_out&MDCR_WDA)==0)   // bits are inverted
                mdcrwritebyte|=0x80;
              mdcrwritebittime=0;
              mdcrbitcounter+=1;
              if((mdcrbitcounter%8)==0) // byte sampled?
              {
                if((mdcrbitcounter/8)<2)  // 0xaa for synchronisation
                  break;
                 else if((mdcrbitcounter/8)<3)  // checksum
                {
                  mdcrchecksum=-mdcrwritebyte;
                  break;
                }
                else if((mdcrbitcounter/8)<259)  // data
                {
                  cassette[mdcrblock*256+mdcrbitcounter/8-3]=mdcrwritebyte;
                  mdcrchecksum+=mdcrwritebyte;
                  break;
                }
                else
                {
                  options|=CASSETTE_CHANGED;
#ifdef MDCRDEBUG
                  if(mdcrchecksum!=0)
                    fprintf(stdout, "cassette position %8d block number %3d checksum error\n", mdcrpos, mdcrblock);
#endif
                  break;  // post 0xaa byte
                }
              }
            }
          }
          break;
        case MDCRFWDSTATE_IDLE :
          if((portd_out&MDCR_WCD)!=0)
          {
            mdcrfwdstate=MDCRFWDSTATE_WRITE;
#ifdef MDCRDEBUG
            fprintf(stdout, "cassette position %8d block number %3d write asserted\n", mdcrpos, mdcrblock);
#endif
          }
          if(mdcrpos>=MDCR_OFFSET)
          {
            if( ((mdcrpos-MDCR_OFFSET)%MDCR_BLOCK) <=5) // max instruction cycle time
            {
              mdcrblock=(mdcrpos-MDCR_OFFSET)/MDCR_BLOCK;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d block id\n", mdcrpos, mdcrblock);
#endif
              mdcrfwdstate=MDCRFWDSTATE_NUMREAD_A;
              mdcrdatapos=0;
              mdcrbitcounter=0;
              break;
            }
          if(mdcrpos>=MDCR_OFFSET+MDCR_DATA)
            if( ((mdcrpos-MDCR_OFFSET-MDCR_DATA)%MDCR_BLOCK) <=5)
            {
              mdcrblock=(mdcrpos-MDCR_OFFSET)/MDCR_BLOCK;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d data begin\n", mdcrpos, mdcrblock);
#endif
              mdcrfwdstate=MDCRFWDSTATE_BLOCKREAD_A;
              mdcrdatapos=0;
              mdcrbitcounter=0;
              mdcrchecksum=0;
              for(i=0; i<256; i++)
                mdcrchecksum+=cassette[256*mdcrblock+i];
              break;
            }
          }
          break;
        case MDCRFWDSTATE_NUMREAD_A:
          if(mdcrdatapos>=(MDCR_BIT/2))
          {
            portc_in|=MDCR_RDC;
            if(mdcrbitcounter>=24)
            {
              mdcrfwdstate=MDCRFWDSTATE_IDLE;
              break;
            }
            portc_in|=MDCR_RDA; // data is inverted
            if( (mdcrbitcounter<8) || (mdcrbitcounter>15) ) 
            {
              if( (0xaa&(1<<(mdcrbitcounter%8))) !=0)
                portc_in&=~MDCR_RDA;
            }
            else
            {
              if( (mdcrblock&(1<<(mdcrbitcounter%8))) !=0)
                portc_in&=~MDCR_RDA;
            }
            mdcrfwdstate=MDCRFWDSTATE_NUMREAD_B;
            mdcrdatapos-=MDCR_BIT/2;
          }
          mdcrdatapos+=cpucycles;
          break;
        case MDCRFWDSTATE_NUMREAD_B:
          if(mdcrdatapos>=(MDCR_BIT/2))
          {
            portc_in&=~MDCR_RDC;                        // data is read by phunsy on falling edge of RDC
            mdcrfwdstate=MDCRFWDSTATE_NUMREAD_A;
            mdcrdatapos-=MDCR_BIT/2;
            mdcrbitcounter++;
          }
          mdcrdatapos+=cpucycles;
          break;
        case MDCRFWDSTATE_BLOCKREAD_A:
          if(mdcrdatapos>=(MDCR_BIT/2))
          {
            portc_in|=MDCR_RDC;
            if(mdcrbitcounter>=2072)
            {
              mdcrfwdstate=MDCRFWDSTATE_IDLE;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d data end\n", mdcrpos, mdcrblock);
#endif
              break;
            }
            portc_in|=MDCR_RDA; // data is inverted
            switch(mdcrbitcounter/8)
            {
              case 0 :
              case 258 :
                if( (0xaa&(1<<(mdcrbitcounter%8))) !=0)
                  portc_in&=~MDCR_RDA;
                break;
              case 1 :
                if( (mdcrchecksum&(1<<(mdcrbitcounter%8))) !=0)
                  portc_in&=~MDCR_RDA;
                break;
              default :
                if( (cassette[256*mdcrblock+mdcrbitcounter/8-2]&(1<<(mdcrbitcounter%8))) !=0)
                  portc_in&=~MDCR_RDA;
                break;
            }
            mdcrfwdstate=MDCRFWDSTATE_BLOCKREAD_B;
            mdcrdatapos-=MDCR_BIT/2;
          }
          mdcrdatapos+=cpucycles;
          break;
        case MDCRFWDSTATE_BLOCKREAD_B:
          if(mdcrdatapos>=(MDCR_BIT/2))
          {
            portc_in&=~MDCR_RDC;                        // data is read by phunsy on falling edge of RDC
            mdcrfwdstate=MDCRFWDSTATE_BLOCKREAD_A;
            mdcrdatapos-=MDCR_BIT/2;
            mdcrbitcounter++;
          }
          mdcrdatapos+=cpucycles;
          break;
      }
      break;
    case MDCRSTATE_REV :
      mdcrsecond+=cpucycles;
      if(mdcrsecond>333333)
      {
#ifdef MDCRDEBUG
        fprintf(stdout, "cassette position %8d\n", mdcrpos);
#endif
        mdcrsecond=0;
      }
      if((portd_out&MDCR_REV)==0) // tape stop
      {
        mdcrstate=MDCRSTATE_IDLE;
        break;
      }
      mdcrpos-=cpucycles;
#ifdef MDCRDEBUG
      if(mdcrpos<MDCR_OFFSET)
        mdcrblock=0;
      else
        mdcrblock=(mdcrpos-MDCR_OFFSET)/MDCR_BLOCK; // actually not required for phunsy mdcr sofware
#endif
// toggle data clock in reverse so the mdcr software can find gaps
      if(mdcrpos<=MDCR_OFFSET)
      {
        mdcrrevstate=MDCRREVSTATE_LGAP;
        portc_in|=MDCR_RDC;
      }
      else
      {
        mdcrblockpos=(mdcrpos-MDCR_OFFSET)%MDCR_BLOCK;
        switch(mdcrrevstate)
        {
          default :
          case MDCRREVSTATE_LGAP :
            if(mdcrblockpos<(MDCR_DATA+MDCR_BLOCKDATLEN))
            {
              mdcrrevstate=MDCRREVSTATE_DATA;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d rev data end  \n", mdcrpos, mdcrblock);
#endif
            }
            break;
          case MDCRREVSTATE_DATA :
            if((mdcrblockpos%56)<28)
              portc_in|=MDCR_RDC;
            else
              portc_in&=~MDCR_RDC;
            if(mdcrblockpos<MDCR_DATA)
            {
              portc_in|=MDCR_RDC;
              mdcrrevstate=MDCRREVSTATE_SGAP;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d rev data begin\n", mdcrpos, mdcrblock);
#endif
            }
            break;
          case MDCRREVSTATE_SGAP :
            if(mdcrblockpos<MDCR_BLOCKNUMLEN)
            {
              mdcrrevstate=MDCRREVSTATE_BNUM;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d rev bnum end  \n", mdcrpos, mdcrblock);
#endif
            }
            break;
          case MDCRREVSTATE_BNUM :
            if((mdcrblockpos%56)<28)
              portc_in|=MDCR_RDC;
            else
              portc_in&=~MDCR_RDC;
            if(mdcrblockpos>(MDCR_DATA+MDCR_BLOCKDATLEN))
            {
              portc_in|=MDCR_RDC;
              mdcrrevstate=MDCRREVSTATE_LGAP;
#ifdef MDCRDEBUG
              fprintf(stdout, "cassette position %8d block number %3d rev bnum begin\n", mdcrpos, mdcrblock);
#endif
            }
            break;
        }
      }
      if(mdcrpos<=0)
      {
        mdcrpos=0;
        mdcrstate=MDCRSTATE_ATSTART;
      }
      break;
    case MDCRSTATE_ATSTART :
      if((portd_out&MDCR_REV)!=0)
        portc_in|=MDCR_BET;
      else
      {
        portc_in&=~MDCR_BET;
        mdcrstate=MDCRSTATE_IDLE;
      }
      break;
    case MDCRSTATE_ATEND :
      if((portd_out&MDCR_FWD)!=0)
        portc_in|=MDCR_BET;
      else
      {
        portc_in&=~MDCR_BET;
        mdcrstate=MDCRSTATE_IDLE;
      }
      break;
  }

#ifdef MDCRDEBUG
  if(mdcrstate!=mdcrstateold)
  {
    fprintf(stdout, "mdcr state %d\n", mdcrstate);
  }
#endif
  mdcrstateold=mdcrstate;

}

void changecassette(int casnum)
{
  int i;

  if((options&CASSETTE_CHANGED)!=0)
    cassettechanged[selectedcassette]=1;
  options&=~CASSETTE_CHANGED;

  if(selectedcassette!=casnum)
    if(mdcrstate==MDCRSTATE_IDLE)
    {
      for(i=0;i<32768;i++)
        cassettes[selectedcassette*32768+i]=cassette[i];
      selectedcassette=casnum;
      for(i=0;i<32768;i++)
        cassette[i]=cassettes[selectedcassette*32768+i];
    }
}


static void wrtc() // bank switching
{
  int i;

  i=(portc_out>>4)&15;
  if(pagelowptr!=i)
  {
    memcpy(&pagelow[pagelowptr*2048],&memory[0x1800],2048);     // store current page
    memcpy(&memory[0x1800],&pagelow[i*2048],2048);              // restore new page
    pagelowptr=i;
  }
  i=portc_out&15;
  if(pagehighptr!=i)
  {
    memcpy(&pagehigh[pagehighptr*16384],&memory[0x4000],16384); // store current page
    memcpy(&memory[0x4000],&pagehigh[i*16384],16384);           // restore new page
    pagehighptr=i;
  }
}


//@@@@@@
/***************************************
** SDL screen driver data and routines
***************************************/

#define COLS	64		// 64 characters per row
#define ROWS	32		// 32 lines per screen
#define CHARX	6		// character cell is 6x8
#define CHARY	8

#define SCALE	2		// and drawing scale is 2x

#define SCRNX	COLS * (CHARX * SCALE)
#define SCRNY	ROWS * (CHARY * SCALE)


static SDL_Window	*window;
static SDL_Renderer	*renderer;
static SDL_Texture	*texture;
static SDL_TimerID	timerID;
static Uint32		*buffer;
static int		buffer_sz;
static Uint32		colormap[8];

static const uint8_t phgraphpix[128] = {
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 0000
  0x38, 0x38, 0x38, 0x38, 0x00, 0x00, 0x00, 0x00,	// 0001
  0x07, 0x07, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00,	// 0010
  0x3f, 0x3f, 0x3f, 0x3f, 0x00, 0x00, 0x00, 0x00,	// 0011
  0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x38, 0x38,	// 0100
  0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,	// 0101
  0x07, 0x07, 0x07, 0x07, 0x38, 0x38, 0x38, 0x38,	// 0110
  0x3f, 0x3f, 0x3f, 0x3f, 0x38, 0x38, 0x38, 0x38,	// 0111
  0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x07, 0x07,	// 1000
  0x38, 0x38, 0x38, 0x38, 0x07, 0x07, 0x07, 0x07,	// 1001
  0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,	// 1010
  0x3f, 0x3f, 0x3f, 0x3f, 0x07, 0x07, 0x07, 0x07,	// 1011
  0x00, 0x00, 0x00, 0x00, 0x3f, 0x3f, 0x3f, 0x3f,	// 1100
  0x38, 0x38, 0x38, 0x38, 0x3f, 0x3f, 0x3f, 0x3f,	// 1101
  0x07, 0x07, 0x07, 0x07, 0x3f, 0x3f, 0x3f, 0x3f,	// 1110
  0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f	// 1111
};


static Uint32
video_blit(Uint32 timer, void *arg)
{
  Uint32 *bufp;
  int pitch;

  if (! SDL_LockTexture(texture, NULL, (void **)&bufp, &pitch))
  {
    (void)pitch;

    memcpy(bufp, buffer, buffer_sz);

    /* Done with the update, unlock and commit. */
    SDL_UnlockTexture(texture);

    /* Update the renderer. */
    SDL_RenderCopy(renderer, texture, NULL, NULL);
    SDL_RenderPresent(renderer);
  }

  return 1;
}


static void
video_write(uint16_t address)
{
  uint8_t ch;
  int col, row, x, y;
  Uint32 color;
  int pitch;
  int offset;
  Uint32 *bufp;
  int pen;

  /* Get the character code from video RAM. */
  ch = memory[address];

  /* Determine offset into video RAM. */
  offset = address - PHSCRADR;
  row = (offset / COLS) * CHARY * SCALE;
  col = (offset % COLS) * CHARX * SCALE;

  bufp = buffer + (row * SCRNX) + col;
  pitch = SCRNX;

  if(ch<128)
  {
    for(y=0;y<CHARY;y++)
    {
      for(x=0;x<CHARX;x++)
      {
        pen = ((charrom[ch*CHARY+y]&(0x20>>x))!=0) ? 7 : 0;
        color = colormap[pen];
       
        bufp[(y*SCALE*pitch)+(x*SCALE)] = color;
        bufp[(y*SCALE*pitch)+(x*SCALE)+1] = color;
        bufp[(((y*SCALE)+1)*pitch)+x*SCALE] = color;
        bufp[(((y*SCALE)+1)*pitch)+(x*SCALE)+1] = color;
      }
    }
  }
  else
  {
    color = colormap[(ch >> 4) & 7];

    for(y=0;y<CHARY;y++)
    {
      for(x=0;x<CHARX;x++)
      {
        if ((phgraphpix[(ch&0x0f)*CHARY+y]&(0x20>>x))!=0)
        {
          bufp[(y*SCALE*pitch)+(x*SCALE)] = color;
          bufp[(y*SCALE*pitch)+(x*SCALE)+1] = color;
          bufp[(((y*SCALE)+1)*pitch)+x*SCALE] = color;
          bufp[(((y*SCALE)+1)*pitch)+(x*SCALE)+1] = color;
        }
        else
        {
          bufp[(y*SCALE*pitch)+(x*SCALE)] = 0;
          bufp[(y*SCALE*pitch)+(x*SCALE)+1] = 0;
          bufp[(((y*SCALE)+1)*pitch)+x*SCALE] = 0;
          bufp[(((y*SCALE)+1)*pitch)+(x*SCALE)+1] = 0;
        }
      }
    }
  }
}


static int
video_init(const char *title)
{
  SDL_PixelFormat *fmt;
  SDL_BlendMode b;
  Uint32 pfmt;
  int i;

  /* Initialize SDL itself. */
  if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER) < 0)
  {
    fprintf(stderr, "Could not initialize SDL: %s\n", SDL_GetError());
    return 1;
  }

  /* Create the SDL window. */
  i = (options & FULLSCREEN) ? SDL_WINDOW_FULLSCREEN_DESKTOP : 0;
  window = SDL_CreateWindow("PHUNSY",
                            SDL_WINDOWPOS_UNDEFINED,
			    SDL_WINDOWPOS_UNDEFINED,
                            SCRNX, SCRNY,
			    i);
  if (window == NULL)
  {
    fprintf(stderr, "Could not initialize video screen: %s\n", SDL_GetError());
    SDL_Quit();
    return 2;
  }

  /* Set the SDL window's title. */
  SDL_SetWindowTitle(window, title);

  /* Get the window's internal pixel format. */
  pfmt = SDL_GetWindowPixelFormat(window);
  fmt = SDL_AllocFormat(pfmt);

  /*
   * Create the colormap we will use.
   *
   * Color 0 is all-black (0,0,0)
   * Color 1..6 are RGB or green-only gradients in steps of (36/256)
   * Color 7 is all-white or all-green
   */
  for (i = 0; i < 7; i++) {
    if (options & GREENSCRN)
      colormap[i]=SDL_MapRGB(fmt, 0, (36*i), 0);
    else
      colormap[i]=SDL_MapRGB(fmt, (36*i), (36*i), (36*i));
  }
  if (options & GREENSCRN)
    colormap[i]=SDL_MapRGB(fmt, 0, 255, 0);
  else
    colormap[i]=SDL_MapRGB(fmt, 255, 255, 255);
  SDL_FreeFormat(fmt);

  /* Create a renderer context. */
  renderer = SDL_CreateRenderer(window, -1, 0);
  SDL_GetRenderDrawBlendMode(renderer, &b);
  if (b != SDL_BLENDMODE_NONE)
    SDL_SetRenderDrawBlendMode(renderer, SDL_BLENDMODE_NONE);

  /* If we are going full-screen, set up the scaler. */
  if (options & FULLSCREEN)
  {
    SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "linear");
    SDL_RenderSetLogicalSize(renderer, SCRNX, SCRNY);
  }

  /* Create the texture we will be writing to. */
  texture = SDL_CreateTexture(renderer,
                              pfmt,
                              SDL_TEXTUREACCESS_STREAMING,
                              SCRNX, SCRNY);
  SDL_GetTextureBlendMode(texture, &b);
  if (b != SDL_BLENDMODE_NONE)
    SDL_SetTextureBlendMode(texture, SDL_BLENDMODE_NONE);

  /* Allocate local texture buffer. */
  buffer_sz = sizeof(Uint32) * (SCRNX * SCRNY);
  buffer = (Uint32 *)malloc(buffer_sz);
  memset(buffer, 0x00, buffer_sz);

  /* Enable the blitter. */
  timerID = SDL_AddTimer(20, video_blit, NULL);

  /* All good! */
  return 0;
}


/* Close down the video driver. */
static void
video_close(void)
{
  SDL_RemoveTimer(timerID);

  if (buffer != NULL)
    free(buffer);

  SDL_Quit();
}


/***************************************
** Emulator
***************************************/

/*
**        output hex string motorola format
**        typ=0 header
**        typ=1 16bit address
**        typ=2 24bit address
**        typ=3 32bit address
**        if len=0, end string is generated
**        len max=250, practical 32
**        returns: length
*/
uint8_t outhexstr(FILE *fp, uint8_t typ, uint8_t len, uint32_t adr, uint8_t *ptr)
{
  uint8_t chk;
  uint8_t c;
  uint8_t i;

  if((typ==0)&&(len==0))
    return(0);                                    /* not allowed          */
  fprintf(fp, "S%1d", len ? typ : 10-typ);
  chk=(uint8_t)(len+typ+2);
  if(typ==0)
    chk++;
  fprintf(fp, "%02X", chk);
  switch(typ)
  {
    case 3 :
      c=(uint8_t)((adr>>24)&0xff);
      chk+=c;
      fprintf(fp, "%02X", c);
    case 2 :
      c=(uint8_t)((adr>>16)&0xff);
      chk+=c;
      fprintf(fp, "%02X", c);
    case 1 :
      c=(uint8_t)((adr>>8)&0xff);
      chk+=c;
      fprintf(fp, "%02X", c);
      c=(uint8_t)(adr&0xff);
      chk+=c;
      fprintf(fp, "%02X", c);
      break;
    case 0 :
      fprintf(fp, "0000");
      break;
  }
  for(i=0; i<len; i++)
  {
    c=ptr[i];
    chk+=c;
    fprintf(fp, "%02X", c);
  }
  fprintf(fp, "%02X\n", ~chk&255);
  return(len);
}

uint32_t gethex(const char *p, int n)
{
  uint32_t h = 0;
  int i = 0;
  char c;
  while((n==0)||(i<n))
  {
    if(!(isxdigit(c=toascii(*p))))
      break;
    if(isdigit(c))
      h = (h<<4)+(c-'0');
    else
      h = (h<<4)+(toupper(c)-'A'+10);
    p++;
    i++;
  }
  return(h);
}


/*
**  input file motorola format
*/
int inphexfile(const char *fname)
{
  char strbuf[250+1];
  FILE *ifp;
  int ret=0;
  int bytecount;
  uint32_t address;
  int strindex;
  int i;
  uint8_t check;

  if((ifp=fopen(fname, "r"))!=NULL)
  {
    while( ((fgets(strbuf, sizeof(strbuf)-1, ifp))!=NULL) && (ret>=0) )
    {
      if(strbuf[0]=='S')
      {
        bytecount=(int)gethex(&strbuf[2],2);
        check=0;
        for(i=0; i<(bytecount+1); i++)
          check+=gethex(&strbuf[i*2+2], 2);
        if(check!=0xff)
          ret=-1;
        else
        {
          switch(strbuf[1])
          {
            case '0' :
            case '1' :
              address=gethex(&strbuf[4], 4);
              strindex=8;
              bytecount-=3;
              break;
            case '2' :
              address=gethex(&strbuf[4], 6);
              strindex=10;
              bytecount-=4;
              break;
            case '3' :
              address=gethex(&strbuf[4], 8);
              strindex=12;
              bytecount-=5;
              break;
            default :
              bytecount=0;
              break;            
          }
          if(bytecount>0)
          {
            if((address+bytecount)>0x7fff)
              ret=-2;
            else
            {
              if(strbuf[1]!='0')
              {
                ret+=bytecount;
                for(i=0; i<bytecount; i++)
                  memory[address+i]=gethex(&strbuf[strindex+i*2], 2);
              }
              else  // S0 can be used for bank selection
                if(strncmp(&strbuf[8], "5048554E5359", 12)==0) // PHUNSY in hex followed by bank byte in hex
                {
                  portc_out=gethex(&strbuf[20], 2);
                  wrtc();
                }
            }
          }
        }
      }
    }
    fclose(ifp);
  }
  return(ret);
}

// this function triggers user-event, so we avoid multi-thread / locking issues.
Uint32 my_callbackfunc(Uint32 interval, void *param)
{
     SDL_Event event;
     SDL_UserEvent userevent;
 
     userevent.type = SDL_USEREVENT;
     userevent.code = 0;
     userevent.data1 = NULL;
     userevent.data2 = NULL;
 
     event.type = SDL_USEREVENT;
     event.user = userevent;
 
     SDL_PushEvent(&event);
     return(interval);
}


int main(int argc, char *argv[])
{
  char temp[1024];
  FILE *fp;
  FILE *logfp = NULL;
  int i, next;
  int ret = 0;
  int ready=0;
  int key;
  int breakpoint;
  int tracelength;
  SDL_Event event;
  SDL_TimerID my_timer_id = -1;

  sprintf(temp, "%s, version %s", str1, APP_VERSION);
  printf("%s\n%s\n", temp, str2);

  options=0;
  memory=(uint8_t *)malloc(32768);
  pagelow=(uint8_t *)malloc(2048*16);
  pagehigh=(uint8_t *)malloc(16384*16);
  cassette=(uint8_t *)malloc(32768);
  cassettes=(uint8_t *)malloc(32768*4);
  extio=(uint8_t *)malloc(128*17);

  for(i=0; i<32768; i++)
  {
    memory[i]=0x40;
    pagelow[i]=0x40;
    cassette[i]=0;
  }
  for(i=0; i<(32768*4); i++)
    cassettes[i]=0;
  for(i=0; i<4; i++)
    cassettechanged[i]=0;

  for(i=0; i<(16384*16); i++)
    pagehigh[i]=0x40;
  for(i=0; i<2048; i++)
  {
    memory[i]=monitor_code[i];
    pagelow[i+1*2048]=mdcr_code[i];
    pagelow[i+2*2048]=dass_code[i];
    pagelow[i+3*2048]=labhnd_code[i];
  }
  options|=INFILE; // indicate code is present
  for(i=0; i<160; i++)
    memory[i+2048]=hello[i];

  pagelowptr=0;
  pagehighptr=0;
  registers.na=0;
  registers.psu&=~II;
  portc_in=PAR_TTYN|BPS_300|MDCR_CIP|MDCR_WEN|MDCR_RDC;      // Select parallel i/o. baudrate less important at this 
  mdcrstate=MDCRSTATE_IDLE;
  mdcrpos=0;                      // set tape to begin position
  portd_in=0xff;                  // indicate no key pressed
  historyindex=0;
  selectedcassette=0;
  ext126_outold=0;

  next=1;
  tracelength=0;
  while(next<argc)
  {
    if(strcmp("-F", argv[next])==0)  // full-screen mode
    {
      next++;
      options |= FULLSCREEN;
      continue;
    }
    if(strcmp("-G", argv[next])==0)  // green-on-black screen mode
    {
      next++;
      options |= GREENSCRN;
      continue;
    }
    if(strcmp("-c", argv[next])==0)  // mdcr file
    {
      if((next+1)<argc)
      {
        next++;
        if((fp=fopen(argv[next], "rb"))!=NULL)
        {
          for(i=0; i<128; i++) // there may be 128 blocks of 256
            fread(&cassette[i*256], 256, 1, fp);
          fclose(fp);
          if(selectedcassette>3)
          {
            fprintf(stderr, "max cassettes is 4\n");
            exit(1);
          }
          for(i=0;i<32768;i++)
            cassettes[selectedcassette*32768+i]=cassette[i];
          selectedcassette++;
        }
        else
          fprintf(stderr, "file read error: %s\n", argv[next]);
      }
      else
        fprintf(stdout, "missing parameters for -c option\n");
      next++;
      continue;
    }
    if(strcmp("-l", argv[next])==0)  // log file
    {
      if((next+1)<argc)
      {
        next++;
        if((logfp=fopen(argv[next], "wb"))!=NULL)
          options|=LOGFILE;
        else
          fprintf(stderr, "can not create: %s\n", argv[next]);
      }
      else
        fprintf(stdout, "missing parameters for -l option\n");
      next++;
      continue;
    }
    if(strcmp("-t", argv[next])==0)
    {
      if((next+2)<argc)
      {
        next++;
        options|=TRACE;
        sscanf(argv[next], "%x", &breakpoint);
        next++;
        sscanf(argv[next], "%d", &tracelength);
        tracelength++;
      }
      else
        fprintf(stdout, "missing parameters for -c option\n");
      next++;
      continue;
    }
    if(strcmp("-h", argv[next])==0)
    {
      next++;
      fprintf(stdout, "usage: phunsy [-F] [-G] -f infile -c cassette -l logfile -t addr length\n");
      fprintf(stdout, "'infile' is in motorola format, multiple -f options allowed\n");
      fprintf(stdout, "'cassette' is a binary file of 32KB with phunsy formatted data\n");
      fprintf(stdout, "'logfile' logs anything written to extended i/o address 7Fh\n");
      fprintf(stdout, "'addr length' can be used for a single trace\n");
      fprintf(stdout, "\nIf -F is specified, Full-Screen mode will be used.\n");
      fprintf(stdout, "If -G is specified, Green will be used as primary color.\n");

      exit(0);
    }
    fprintf(stdout, "unknown option %s\n", argv[next]);
    break;
  }

  selectedcassette=0;
  for(i=0;i<32768;i++)
    cassette[i]=cassettes[i];

  /* Initialize SDL driver. */
  if ((i = video_init(temp)) != 0)
	exit(i);

  while(ready==0)
  {
    if((options&TRACE)!=0)
    {
      if(registers.ap==(uint16_t)breakpoint)
        options|=PRINT;
      if((options&PRINT)!=0)
      {
        if(tracelength==0)
          options&=~(PRINT|TRACE);
        else
          tracelength--;
      }
    }
    ret=cpu();
    if((options&PRINT)!=0)
      disassemble();
    memcpy(&history[historyindex++], &registers, sizeof(struct regset2650));
    if(historyindex>=HISTORY_SIZE)
      historyindex=0;
    mdcr_exec();

    if( SDL_PollEvent( &event ) )
    {
      switch( event.type )
      {
        /* Keyboard event */
        /* Pass the event data onto PrintKeyInfo() */
        case SDL_KEYDOWN:
          key=127;
          switch(event.key.keysym.sym)
          {
            case SDLK_RIGHT :
              key=28;
              break;
            case SDLK_LEFT :
              key=29;
              break;
            case SDLK_DOWN :
              key=30;
              break;
            case SDLK_UP :
              key=31;
              break;
            case SDLK_F1 :
              changecassette(0);
              break;
            case SDLK_F2 :
              changecassette(1);
              break;
            case SDLK_F3 :
              changecassette(2);
              break;
            case SDLK_F4 :
              changecassette(3);
              break;
            case SDLK_F12 :
              ready = 1;
              break;
            case '\b':
              key = 8;
              break;
            case '\r':
              key = 13;
              break;
            default :
              /* It will be handled through the TEXTINPUT event. */
              break;
          }
          if(key!=127)
            portd_in=(uint8_t)key;
          break;

        case SDL_TEXTINPUT:
          key = 127;
          if(event.text.text[0]>0)
            key=event.text.text[0]&0x7f;
          if(key!=127)
            portd_in=(uint8_t)key;
          break;

        case SDL_USEREVENT:
          if((registers.psu&II)==0)
            interrupt();
          break;      	

        case SDL_QUIT:
          ready=1;
          break;

        default:
          break;
      }
    }

    if((portd_out&KEYB_ACK)!=0)
      portd_in|=0x80;

    switch(ret&0xffff0000)
    {
#if 0
      case NORMAL : // this handles i/o for now...
        if((ret&0x7fff)==0x05fa) // wchar - capture character to screen
        {
          if(registers.r[0]!=0x0a)
            fputc(registers.r[0], stdout);
          if(registers.r[0]==0x0d)
            fputc(0x0a, stdout);
        }
        break;
#endif

      case PC_WRITE : // bank switching
        wrtc();
        break;

      case HALT :
        fprintf(stdout, "halt instruction encountered at address %04X\n", ret&0xffff);
        ready=ret;
        break;

      case INV_OPCODE :
        fprintf(stdout, "invalid instruction encountered at address %04X\n", ret&0xffff);
        ready=ret;
        break;

      case EXT_WRITE :
        if((options&LOGFILE)!=0) {
          if((ret&255)==127)
            fputc(extio[127], logfp);
        }
        if((ret&255)==127) {
          if(extio[126]!=ext126_outold)
          {
            if(extio[126]==0)
            {
              if (my_timer_id != -1)
                SDL_RemoveTimer(my_timer_id);
              my_timer_id = -1;
            }
            else
            {
              if (my_timer_id != -1)
                SDL_RemoveTimer(my_timer_id);
              my_timer_id = SDL_AddTimer(10*extio[126], my_callbackfunc, 0);
            }
          }
        }
        break;

      case MEM_WRITE :
        if( ((ret&0x7fff)>=0x1000) && ((ret&0x7fff)<0x1800) )
          video_write(ret&0x7fff);
        else if((ret&0x7fff)<0x0800)
        {
          fprintf(stdout, "write to monitor rom at address %04X\n", ret&0xffff);
          ready=ret;
        }
        else if( ((ret&0x7fff)>=0x1800) && ((ret&0x7fff)<0x1fff) && ( ((portc_out&0xf0)==0x10) || ((portc_out&0xf0)==0x20) || ((portc_out&0xf0)==0x30) ) )
        {
          fprintf(stdout, "write to mdcr rom at address %04X\n", ret&0xffff);
          ready=ret;
        }
        break;

#if 0
      case MEM_READ :
      case PD_WRITE :
      case PC_READ :
      case PD_READ :
      case EXT_READ :
#endif
      default :
        break;
    }
  }

  if((options&LOGFILE)!=0)
    fclose(logfp);

  for(i=0;i<4;i++)
  {
    changecassette(i);
    if(cassettechanged[i]!=0)
    {
      sprintf(temp, "c%d.mdcr", i+1);
      if((fp=fopen(temp, "wb"))==NULL)
        fprintf(stdout, "can't write %s\n", temp);
      else
      {
        fwrite(cassette, 1, 32768, fp);
        fclose(fp);
      }
    }
  }

  if((fp=fopen("memory.raw", "wb"))==NULL)
    fprintf(stdout, "can't write memory.raw\n");
  else
  {
    fwrite(memory, 1, 32768, fp);
    fclose(fp);
  }

/*
  if((fp=fopen("memoryQ.raw", "wb"))==NULL)
    fprintf(stdout, "can't write memory.raw\n");
  else
  {
    fwrite(pagehigh, 16, 16384, fp);
    fclose(fp);
  }
*/

  if( ((ret&0xffff0000)==HALT) || ((ret&0xffff0000)==INV_OPCODE) || ((ret&0xffff0000)==MEM_WRITE) )
  {
    options|=PRINT;
    for(i=0; i<HISTORY_SIZE; i++)
    {
      memcpy(&registers, &history[historyindex++], sizeof(struct regset2650));
      if(historyindex>=HISTORY_SIZE)
        historyindex=0;
      disassemble();
    }
  }

  video_close();

  return 0;
}
